<?php

namespace App\Controllers;

use App\Models\CustomerModel;

class CustomersController extends BaseController
{
    /**
     * Display customers list with search & pagination
     */
    public function index()
{
    $model = new CustomerModel();

    // Get search query
    $search = trim($this->request->getGet('search') ?? '');

    // Apply search filter if any
    if ($search !== '') {
        $model->groupStart()
              ->like('name', $search)
              ->orLike('phone', $search)
              ->orLike('address', $search)
              ->groupEnd();
    }

    // Get all customers without pagination
    $customers = $model->orderBy('id', 'DESC')->findAll(); // ✅ no pagination

    $data = [
        'customers' => $customers ?: [], // always an array
        'search'    => $search,
    ];

    return view('sales/customers', $data);
}


    /**
     * Store one or multiple customers
     */
    public function store()
{
    $names     = array_map('trim', (array) $this->request->getPost('name'));
    $phones    = array_map('trim', (array) $this->request->getPost('phone'));
    $addresses = array_map('trim', (array) $this->request->getPost('address'));

    $model = new CustomerModel();
    $errors = [];

    foreach ($names as $i => $name) {
        if ($name === '') continue;

        $validation = \Config\Services::validation();
        $validation->setRules([
            "name" => 'required|min_length[2]',
            "phone" => 'permit_empty',
            "address" => 'permit_empty',
        ]);

        $validData = [
            'name'    => $name,
            'phone'   => $phones[$i] ?? null,
            'address' => $addresses[$i] ?? null,
        ];

        if (!$validation->run($validData)) {
            $errors[] = "Row " . ($i + 1) . ": " . implode(", ", $validation->getErrors());
            continue;
        }

        $model->insert($validData);
    }

    if (!empty($errors)) {
        return redirect()->back()
            ->withInput()
            ->with('errors', $errors);
    }

    return redirect()->to(site_url('customers'))
        ->with('success', 'Customer(s) added successfully.');
}


    /**
     * Delete customer
     */
    public function delete($id)
    {
        $model = new CustomerModel();

        if (!$model->find($id)) {
            return redirect()->to(site_url('customers'))
                ->with('errors', 'Customer not found.');
        }

        $model->delete($id);

        return redirect()->to(site_url('customers'))
            ->with('success', 'Customer deleted successfully.');
    }

    /**
     * View customer (AJAX)
     */
    public function view($id)
    {
        $model = new CustomerModel();
        $customer = $model->find($id);

        if (!$customer) {
            return $this->response
                ->setStatusCode(404)
                ->setJSON(['error' => 'Customer not found']);
        }

        return $this->response->setJSON($customer);
    }

    /**
     * Update customer
     */
    public function update()
    {
        $id = $this->request->getPost('id');
        $model = new CustomerModel();

        if (!$model->find($id)) {
            return redirect()->to(site_url('customers'))
                ->with('errors', 'Customer not found.');
        }

        $data = [
            'name'    => $this->request->getPost('name'),
            'phone'   => $this->request->getPost('phone'),
            'address' => $this->request->getPost('address'),
        ];

        $validation = \Config\Services::validation();
        $validation->setRules([
            'name' => 'required|min_length[2]',
            'phone' => 'permit_empty',
            'address' => 'permit_empty',
        ]);

        if (!$validation->run($data)) {
            return redirect()->back()
                ->withInput()
                ->with('errors', $validation->getErrors());
        }

        $model->update($id, $data);

        return redirect()->to(site_url('customers'))
            ->with('success', 'Customer updated successfully.');
    }
}
