<?php

namespace App\Controllers;

use App\Models\PaymentModel;
use App\Models\VendorModel;
use App\Models\PurchaseModel;
use CodeIgniter\Controller;
use Dompdf\Dompdf;

class PaymentController extends Controller
{
    protected $paymentModel;
    protected $vendorModel;
    protected $purchaseModel;

    public function __construct()
    {
        $this->paymentModel  = new PaymentModel();
        $this->vendorModel   = new VendorModel();
        $this->purchaseModel = new PurchaseModel();
        helper(['form', 'url']);
    }

    /**
     * Display payments with search and pagination
     */
    public function index()
    {
        $search = $this->request->getGet('search');

        $paymentsQuery = $this->paymentModel
            ->select('payments.*, vendors.name AS vendor_name')
            ->join('vendors', 'vendors.id = payments.vendor_id')
            ->orderBy('payment_date', 'DESC');

        if ($search) {
            $paymentsQuery->groupStart()
                ->like('vendors.name', $search)
                ->orLike('payments.amount', $search)
                ->orLike('payments.payment_date', $search)
                ->orLike('payments.note', $search)
            ->groupEnd();
        }

        $payments = $paymentsQuery->paginate(10);
        $pager = $this->paymentModel->pager;

        $vendors = $this->vendorModel->orderBy('name', 'ASC')->findAll();
        foreach ($vendors as &$v) {
            $purchaseRow = $this->purchaseModel
                ->select('SUM(price * quantity) AS total')
                ->where('vendor_name', $v['name'])
                ->first();
            $totalPurchases = $purchaseRow['total'] ?? 0;

            $paymentRow = $this->paymentModel
                ->select('SUM(amount) AS paid')
                ->where('vendor_id', $v['id'])
                ->first();
            $totalPaid = $paymentRow['paid'] ?? 0;

            $v['total_purchases'] = $totalPurchases;
            $v['total_paid']      = $totalPaid;
            $v['balance']         = $totalPurchases - $totalPaid;
        }

        return view('payments/index', [
            'vendors'  => $vendors,
            'payments' => $payments,
            'pager'    => $pager,
            'success'  => session()->getFlashdata('success'),
            'search'   => $search
        ]);
    }

    /**
     * Store a new payment
     */
    public function store()
    {
        $this->paymentModel->insert([
            'vendor_id'    => $this->request->getPost('vendor_id'),
            'amount'       => $this->request->getPost('amount'),
            'payment_date' => $this->request->getPost('payment_date') ?: date('Y-m-d'),
            'note'         => $this->request->getPost('note'),
        ]);

        return redirect()->to('/payments')->with('success', 'Payment recorded.');
    }

    /**
     * Delete a payment
     */
    public function delete($id)
    {
        $this->paymentModel->delete($id);
        return redirect()->back()->with('success', 'Payment deleted.');
    }

    /**
     * Download PDF of payments (NO LOGO)
     */
    public function downloadPdf()
    {
        $payments = $this->paymentModel
            ->select('payments.*, vendors.name AS vendor_name')
            ->join('vendors', 'vendors.id = payments.vendor_id')
            ->orderBy('payment_date', 'DESC')
            ->findAll();

        $html = '
            <h2 style="text-align:center; margin-bottom:20px;">
                Vendor Payments
            </h2>

            <table border="1" width="100%" cellpadding="6" cellspacing="0" style="border-collapse:collapse;">
                <tr style="background:#f0f0f0;">
                    <th>Vendor</th>
                    <th>Amount</th>
                    <th>Date</th>
                    <th>Note</th>
                </tr>
        ';

        foreach ($payments as $p) {
            $html .= '
                <tr>
                    <td>' . esc($p['vendor_name']) . '</td>
                    <td>' . number_format($p['amount'], 2) . '</td>
                    <td>' . esc($p['payment_date']) . '</td>
                    <td>' . esc($p['note']) . '</td>
                </tr>
            ';
        }

        $html .= '</table>';

        $dompdf = new Dompdf();
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        return $dompdf->stream('vendor_payments.pdf', ['Attachment' => true]);
    }

    /**
     * View purchases and payments for a specific vendor
     */
    public function vendorItems($vendorId)
{
    $vendor = $this->vendorModel->find($vendorId);
    if (!$vendor) {
        return redirect()->back()->with('error', 'Vendor not found');
    }

    // Fetch purchases with item names
    $purchases = $this->purchaseModel
    ->where('vendor_name', $vendor['name'])
    ->orderBy('created_at', 'ASC')
    ->findAll();

    $payments = $this->paymentModel
        ->where('vendor_id', $vendorId)
        ->orderBy('payment_date', 'ASC')
        ->findAll();

    return view('vendors/items', [
        'vendor'    => $vendor,
        'purchases' => $purchases,
        'payments'  => $payments,
    ]);
}

}
