<?php

namespace App\Controllers;

use App\Models\ProductModel;
use CodeIgniter\Controller;

class ProductsController extends Controller
{
    protected $productModel;

    public function __construct()
    {
        $this->productModel = new ProductModel();
        helper(['form', 'url']);
    }

    /**
     * List products with search and pagination
     */
    public function index()
    {
        if (!session()->get('logged_in')) {
            return redirect()->to('/login');
        }

        $search  = $this->request->getGet('search');
        $builder = $this->productModel;

        if (!empty($search)) {
            $builder->like('name', $search);
        }

        $products = $builder
            ->orderBy('id', 'DESC')
            ->paginate(10);

        return view('products/index', [
            'products' => $products,
            'pager'    => $this->productModel->pager,
            'search'   => $search,
            'success'  => session()->getFlashdata('success'),
            'errors'   => session()->getFlashdata('errors'),
        ]);
    }

    /**
     * Store multiple products (name + price only)
     */
    public function store()
    {
        $names  = (array) $this->request->getPost('name');
        $prices = (array) $this->request->getPost('price');

        $errors = [];

        foreach ($names as $index => $name) {

            // Skip empty rows
            if (trim($name) === '' && trim($prices[$index]) === '') {
                continue;
            }

            if (strlen(trim($name)) < 2) {
                $errors[] = "Row " . ($index + 1) . ": Product name must be at least 2 characters.";
            }

            if (!is_numeric($prices[$index])) {
                $errors[] = "Row " . ($index + 1) . ": Price must be a numeric value.";
            }
        }

        if (!empty($errors)) {
            return redirect()
                ->to('/products')
                ->with('errors', $errors)
                ->withInput();
        }

        // Insert products
        foreach ($names as $index => $name) {

            if (trim($name) === '' && trim($prices[$index]) === '') {
                continue;
            }

            $this->productModel->insert([
                'name'  => trim($name),
                'price' => $prices[$index],
            ]);
        }

        return redirect()
            ->to('/products')
            ->with('success', 'Products added successfully.');
    }

    /**
     * Delete product
     */
    public function delete($id)
    {
        $product = $this->productModel->find($id);

        if (!$product) {
            return redirect()
                ->to('/products')
                ->with('errors', ['Product not found.']);
        }

        $this->productModel->delete($id);

        return redirect()
            ->to('/products')
            ->with('success', 'Product deleted successfully.');
    }

    /**
     * View product details (AJAX)
     */
    public function view($id)
    {
        $product = $this->productModel->find($id);

        if (!$product) {
            return $this->response
                ->setStatusCode(404)
                ->setJSON(['error' => 'Product not found']);
        }

        return $this->response->setJSON([
            'id'         => $product['id'],
            'name'       => $product['name'],
            'price'      => $product['price'],
            'created_at' => $product['created_at'] ?? date('Y-m-d H:i:s'),
            'updated_at' => $product['updated_at'] ?? date('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Edit product
     */
    public function edit($id)
    {
        $product = $this->productModel->find($id);

        if (!$product) {
            return redirect()
                ->to('/products')
                ->with('errors', ['Product not found.']);
        }

        return view('products/edit', [
            'product' => $product,
            'errors'  => session()->getFlashdata('errors'),
        ]);
    }

    /**
     * Update product
     */
    public function update($id)
    {
        $product = $this->productModel->find($id);

        if (!$product) {
            return redirect()
                ->to('/products')
                ->with('errors', ['Product not found.']);
        }

        $validation = \Config\Services::validation();
        $validation->setRules([
            'name'  => 'required|min_length[2]|max_length[150]',
            'price' => 'required|numeric',
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()
                ->back()
                ->withInput()
                ->with('errors', $validation->getErrors());
        }

        $this->productModel->update($id, [
            'name'  => $this->request->getPost('name'),
            'price' => $this->request->getPost('price'),
        ]);

        return redirect()
            ->to('/products')
            ->with('success', 'Product updated successfully.');
    }
}
