<?php

namespace App\Controllers;

use App\Models\PurchaseModel;
use App\Models\VendorModel;
use App\Models\ProductModel;
use CodeIgniter\Controller;

class PurchaseController extends Controller
{
    protected $purchaseModel;
    protected $vendorModel;
    protected $productModel;

    public function __construct()
    {
        $this->purchaseModel = new PurchaseModel();
        $this->vendorModel   = new VendorModel();
        $this->productModel  = new ProductModel();
        helper(['form', 'url', 'text']);
    }

    // =========================
    // List Purchases with Search & Pagination
    // =========================
    public function index()
    {
        $search = $this->request->getGet('search') ?? '';
        $page   = $this->request->getGet('page') ?? 1;

        $builder = $this->purchaseModel;

        if ($search) {
            $builder->groupStart()
                    ->like('vendor_name', $search)
                    ->orLike('product_name', $search)
                    ->orLike('purchase_date', $search)
                    ->groupEnd();
        }

        $perPage = 10; // items per page
        $purchases = $builder->orderBy('purchase_date', 'DESC')->paginate($perPage, 'default', $page);
        $pager     = $builder->pager;

        return view('purchases/index', [
            'purchases' => $purchases,
            'pager'     => $pager,
            'search'    => $search,
            'success'   => session()->getFlashdata('success'),
            'vendors'   => $this->vendorModel->orderBy('name', 'ASC')->findAll(),
            'products'  => $this->productModel->orderBy('name', 'ASC')->findAll(),
        ]);
    }

    // =========================
    // Store Purchases
    // =========================
    public function store()
    {
        $vendorsSelected  = $this->request->getPost('vendor') ?? [];
        $productsSelected = $this->request->getPost('product') ?? [];
        $vendorsNew       = $this->request->getPost('vendor_new') ?? [];
        $productsNew      = $this->request->getPost('product_new') ?? [];
        $prices           = $this->request->getPost('price') ?? [];
        $qtys             = $this->request->getPost('quantity') ?? [];
        $globalDate       = $this->request->getPost('purchase_date_global') ?: date('Y-m-d');

        foreach ($prices as $i => $price) {

            // Vendor
            if (!empty($vendorsSelected[$i]) && $vendorsSelected[$i] === '_new') {
                $vendorName = trim($vendorsNew[$i]);
                if ($vendorName) $this->vendorModel->insert(['name' => $vendorName]);
            } else {
                $vendorName = $vendorsSelected[$i] ?? '';
            }

            // Product
            if (!empty($productsSelected[$i]) && $productsSelected[$i] === '_new') {
                $productName = trim($productsNew[$i]);
                if ($productName) $this->productModel->insert(['name' => $productName]);
            } else {
                $productName = $productsSelected[$i] ?? '';
            }

            if (!$vendorName || !$productName) continue;

            $this->purchaseModel->insert([
                'vendor_name'   => $vendorName,
                'product_name'  => $productName,
                'price'         => $price,
                'quantity'      => $qtys[$i],
                'purchase_date' => $globalDate,
            ]);
        }

        return redirect()->to('/purchases')->with('success', 'Purchase(s) recorded successfully.');
    }

    // =========================
    // Edit Purchase
    // =========================
    public function edit($id)
    {
        $purchase = $this->purchaseModel->find($id);

        if (!$purchase) {
            return redirect()->to('/purchases')->with('error', 'Purchase not found.');
        }

        return view('purchases/edit', [
            'purchase' => $purchase,
            'vendors'  => $this->vendorModel->orderBy('name', 'ASC')->findAll(),
            'products' => $this->productModel->orderBy('name', 'ASC')->findAll(),
        ]);
    }

    // =========================
    // Update Purchase
    // =========================
    public function update($id)
    {
        $vendorName   = $this->request->getPost('vendor');
        $productName  = $this->request->getPost('product');
        $price        = $this->request->getPost('price');
        $quantity     = $this->request->getPost('quantity');
        $purchaseDate = $this->request->getPost('purchase_date') ?: date('Y-m-d');

        $this->purchaseModel->update($id, [
            'vendor_name'   => $vendorName,
            'product_name'  => $productName,
            'price'         => $price,
            'quantity'      => $quantity,
            'purchase_date' => $purchaseDate,
        ]);

        return redirect()->to('/purchases')->with('success', 'Purchase updated successfully.');
    }

    // =========================
    // Delete Purchase
    // =========================
    public function delete($id)
    {
        $this->purchaseModel->delete($id);
        return redirect()->back()->with('success', 'Purchase deleted.');
    }
}
