<?php 

namespace App\Controllers;

use App\Models\SaleModel;
use App\Models\SalesPaymentModel;
use App\Models\MyDailyMoneyModel;
use App\Models\PurchaseModel;
use App\Models\PaymentModel;

class Reports extends BaseController
{
    protected $saleModel;
    protected $paymentModel;
    protected $moneyModel;
    protected $purchaseModel;
    protected $purchasePaymentModel;

    public function __construct()
    {
        $this->saleModel            = new SaleModel();
        $this->paymentModel         = new SalesPaymentModel();
        $this->moneyModel           = new MyDailyMoneyModel(); // for profit.php
        $this->purchaseModel        = new PurchaseModel();
        $this->purchasePaymentModel = new PaymentModel();
    }

    // Reports Dashboard
    public function index()
    {
        $today = date('Y-m-d');

        // Profit & Loss Card
        $todayIncome = $this->moneyModel
            ->where('type', 'income')
            ->where('DATE(created_at)', $today)
            ->selectSum('amount')
            ->first()['amount'] ?? 0;

        $todayExpense = $this->moneyModel
            ->where('type', 'expense')
            ->where('DATE(created_at)', $today)
            ->selectSum('amount')
            ->first()['amount'] ?? 0;

        // Sales Card
        $todaySales = $this->saleModel
            ->where("DATE(created_at) =", $today)
            ->findAll();

        $todayPayments = $this->paymentModel
            ->where("DATE(created_at) =", $today)
            ->selectSum('amount')
            ->first()['amount'] ?? 0;

        $todaySalesTotal = 0;
        foreach ($todaySales as $sale) {
            $todaySalesTotal += $sale['quantity'] * $sale['price'];
        }

        // Purchases Card
        $todayPurchases = $this->purchaseModel
            ->where("DATE(created_at) =", $today)
            ->findAll();

        $todayPurchasePayments = $this->purchasePaymentModel
            ->where("DATE(payment_date) =", $today)
            ->selectSum('amount')
            ->first()['amount'] ?? 0;

        $todayPurchasesTotal = 0;
        foreach ($todayPurchases as $purchase) {
            $todayPurchasesTotal += $purchase['quantity'] * $purchase['price'];
        }

        return view('reports/index', [
            // Profit & Loss
            'todayIncome'      => $todayIncome,
            'todayExpense'     => $todayExpense,
            'todayBalance'     => $todayIncome - $todayExpense,

            // Sales
            'todaySalesTotal'  => $todaySalesTotal,
            'todayPayments'    => $todayPayments,

            // Purchases
            'todayPurchasesTotal'        => $todayPurchasesTotal,
            'todayPurchasePayments'      => $todayPurchasePayments,
        ]);
    }

    // Sales Report Page
    public function sales()
    {
        $today      = date('Y-m-d');
        $weekStart  = date('Y-m-d', strtotime('monday this week'));
        $monthStart = date('Y-m-01');
        $yearStart  = date('Y-01-01');

        // Sales totals (quantity × price)
        $salesTotal = function ($from) {
            $rows = $this->saleModel
                ->where('created_at >=', $from)
                ->findAll();

            $total = 0;
            foreach ($rows as $row) {
                $total += ((float)$row['quantity'] * (float)$row['price']);
            }
            return $total;
        };

        // Payments totals
        $paymentsTotal = function ($from) {
            return $this->paymentModel
                ->where('created_at >=', $from)
                ->selectSum('amount')
                ->first()['amount'] ?? 0;
        };

        return view('reports/sales', [
            'dailySales'    => $salesTotal($today),
            'weeklySales'   => $salesTotal($weekStart),
            'monthlySales'  => $salesTotal($monthStart),
            'yearlySales'   => $salesTotal($yearStart),

            'dailyPayments'    => $paymentsTotal($today),
            'weeklyPayments'   => $paymentsTotal($weekStart),
            'monthlyPayments'  => $paymentsTotal($monthStart),
            'yearlyPayments'   => $paymentsTotal($yearStart),
        ]);
    }

    // Alias for interface link
    public function salesReport()
    {
        return $this->sales(); // call sales() logic
    }

    // Profit & Loss Page
    public function profit()
    {
        $today      = date('Y-m-d');
        $weekStart  = date('Y-m-d', strtotime('monday this week'));
        $monthStart = date('Y-m-01');
        $yearStart  = date('Y-01-01');

        $sum = function($type, $from) {
            return $this->moneyModel->where('type', $type)
                                    ->where('created_at >=', $from)
                                    ->selectSum('amount')
                                    ->first()['amount'] ?? 0;
        };

        $data = [
            'dailyIncome'   => $sum('income', $today),
            'dailyExpense'  => $sum('expense', $today),
            'weeklyIncome'  => $sum('income', $weekStart),
            'weeklyExpense' => $sum('expense', $weekStart),
            'monthlyIncome' => $sum('income', $monthStart),
            'monthlyExpense'=> $sum('expense', $monthStart),
            'yearlyIncome'  => $sum('income', $yearStart),
            'yearlyExpense' => $sum('expense', $yearStart),
            'records'       => $this->moneyModel->orderBy('created_at', 'DESC')->findAll()
        ];

        return view('reports/profit', $data);
    }

    // Purchases Report Page
    public function purchases()
    {
        $today      = date('Y-m-d');
        $weekStart  = date('Y-m-d', strtotime('monday this week'));
        $monthStart = date('Y-m-01');
        $yearStart  = date('Y-01-01');

        // Purchases totals
        $calcPurchases = function($from) {
            $rows = $this->purchaseModel
                        ->where('created_at >=', $from)
                        ->findAll();
            $total = 0;
            foreach ($rows as $row) {
                $total += $row['quantity'] * $row['price'];
            }
            return $total;
        };

        // Payments totals
        $calcPayments = function($from) {
            return $this->purchasePaymentModel
                        ->where('payment_date >=', $from)
                        ->selectSum('amount')
                        ->first()['amount'] ?? 0;
        };

        return view('reports/purchases', [
            'dailyPurchases'     => $calcPurchases($today),
            'weeklyPurchases'    => $calcPurchases($weekStart),
            'monthlyPurchases'   => $calcPurchases($monthStart),
            'yearlyPurchases'    => $calcPurchases($yearStart),

            'dailyPayments'      => $calcPayments($today),
            'weeklyPayments'     => $calcPayments($weekStart),
            'monthlyPayments'    => $calcPayments($monthStart),
            'yearlyPayments'     => $calcPayments($yearStart),
        ]);
    }
}
