<?php

namespace App\Controllers;

use App\Models\SaleModel;
use App\Models\CustomerModel;
use App\Models\ProductModel;
use App\Models\PurchaseModel;

class SalesController extends BaseController
{
    protected $customerModel;
    protected $productModel;
    protected $purchaseModel;

    public function __construct()
    {
        $this->customerModel = new CustomerModel();
        $this->productModel  = new ProductModel();
        $this->purchaseModel = new PurchaseModel();
    }

    /**
     * List sales with search (no pagination)
     */
    public function index()
    {
        $search = $this->request->getGet('search');

        $salesModel = new SaleModel();
        $salesModel->select('sales.*, customers.name AS customer_name, products.name AS product_name')
                   ->join('customers', 'customers.id = sales.customer_id', 'left')
                   ->join('products', 'products.id = sales.product_id', 'left')
                   ->orderBy('sales.created_at', 'DESC');

        if (!empty($search)) {
            $salesModel->groupStart()
                       ->like('customers.name', $search)
                       ->orLike('products.name', $search)
                       ->orLike('sales.id', $search)
                       ->groupEnd();
        }

        // Fetch all sales without pagination
        $sales = $salesModel->findAll(); // <-- no paginate

        // Prepare products with available stock
        $products = $this->productModel->findAll();
        foreach ($products as &$product) {
            $totalPurchased = $this->purchaseModel
                                   ->selectSum('quantity')
                                   ->where('product_name', $product['name'])
                                   ->first()['quantity'] ?? 0;

            $totalSold = (new SaleModel())
                         ->selectSum('quantity')
                         ->where('product_id', $product['id'])
                         ->first()['quantity'] ?? 0;

            $product['quantity'] = $totalPurchased - $totalSold;
        }

        return view('sales/index', [
            'sales'     => $sales,
            'search'    => $search,
            'customers' => $this->customerModel->findAll(),
            'products'  => $products,
        ]);
    }

    /**
     * Store new sales
     */
    public function store()
    {
        $post = $this->request->getPost();

        if (empty($post['product_id']) || !is_array($post['product_id'])) {
            return redirect()->back()->withInput()->with('error', 'Please add at least one product.');
        }

        if (empty($post['customer_id'])) {
            return redirect()->back()->withInput()->with('error', 'Customer is required.');
        }

        $errors = [];
        $insertData = [];

        foreach ($post['product_id'] as $index => $productId) {
            $quantity = (int)($post['quantity'][$index] ?? 0);
            $price    = (float)($post['price'][$index] ?? 0);

            if ($quantity <= 0 || $price <= 0) {
                $errors[] = 'Row ' . ($index + 1) . ' has invalid quantity or price.';
                continue;
            }

            $product = $this->productModel->find($productId);
            if (!$product) {
                $errors[] = 'Row ' . ($index + 1) . ': product not found.';
                continue;
            }

            $totalPurchased = $this->purchaseModel
                                   ->selectSum('quantity')
                                   ->where('product_name', $product['name'])
                                   ->first()['quantity'] ?? 0;

            $totalSold = (new SaleModel())
                         ->selectSum('quantity')
                         ->where('product_id', $productId)
                         ->first()['quantity'] ?? 0;

            $availableStock = $totalPurchased - $totalSold;

            if ($quantity > $availableStock) {
                $errors[] = "Row " . ($index + 1) . " exceeds stock ({$availableStock}).";
                continue;
            }

            $insertData[] = [
                'customer_id' => $post['customer_id'],
                'product_id'  => $productId,
                'quantity'    => $quantity,
                'price'       => $price,
                'total'       => $quantity * $price,
                'created_at'  => date('Y-m-d H:i:s'),
            ];
        }

        if ($insertData) {
            (new SaleModel())->insertBatch($insertData);
        }

        $message = 'Sales added successfully.';
        if ($errors) {
            $message .= ' Some rows failed: ' . implode(', ', $errors);
        }

        return redirect()->to(site_url('sales'))->with('success', $message);
    }

    /**
     * Delete a sale
     */
    public function delete($id)
    {
        (new SaleModel())->delete($id);
        return redirect()->to(site_url('sales'))->with('success', 'Sale deleted successfully.');
    }

    /**
     * View sale details (AJAX)
     */
    public function view($id)
    {
        $sale = (new SaleModel())
            ->select('sales.*, customers.name AS customer_name, products.name AS product_name')
            ->join('customers', 'customers.id = sales.customer_id', 'left')
            ->join('products', 'products.id = sales.product_id', 'left')
            ->find($id);

        if (!$sale) {
            return $this->response
                        ->setStatusCode(404)
                        ->setJSON(['error' => 'Sale not found']);
        }

        return $this->response->setJSON($sale);
    }

    // ... keep the rest of your methods (generateInvoice, latestInvoice, downloadInvoice, edit) as-is
public function latestInvoice()
    {
        $sale = (new SaleModel())
            ->select('sales.*, customers.name AS customer_name, customers.phone, customers.address, products.name AS product_name')
            ->join('customers', 'customers.id = sales.customer_id', 'left')
            ->join('products', 'products.id = sales.product_id', 'left')
            ->orderBy('sales.created_at', 'DESC')
            ->findAll(1);

        if (!$sale) {
            return redirect()->back()->with('error', 'No sales found.');
        }

        $customers = $this->customerModel->findAll();

        return view('sales/generate_invoice', [
            'sale' => $sale,
            'customers' => $customers,
        ]);
    }

    public function downloadInvoice($id)
    {
        $saleModel = new \App\Models\SaleModel();
        $firstSale = $saleModel->getSaleById($id);

        if (empty($firstSale)) {
            return redirect()->back()->with('error', 'Invoice not found.');
        }

        $firstSale = $firstSale[0];
        $sales = $saleModel->getInvoiceSales($firstSale['customer_id'], $firstSale['created_at']);

        if (empty($sales)) {
            return redirect()->back()->with('error', 'No invoice items found.');
        }

        $data = [
            'sale' => $sales,
            'transport_cost' => (float) ($this->request->getGet('transport_cost') ?? 0),
            'tax' => $this->request->getGet('tax') == 1
        ];

        $dompdf = new \Dompdf\Dompdf([
            'isRemoteEnabled' => true,
            'isHtml5ParserEnabled' => true
        ]);

        $html = view('sales/generate_invoice_pdf', $data);
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        return $dompdf->stream("Invoice_{$id}.pdf", ['Attachment' => true]);
    }

    public function edit($id)
    {
        $saleModel = new \App\Models\SaleModel();
        $saleData = $saleModel->getSaleById($id);

        if (empty($saleData)) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound("Sale not found");
        }

        $data['sale'] = $saleData[0];
        $data['customers'] = $this->customerModel->findAll();
        $data['products'] = $this->productModel->findAll();

        return view('sales/edit', $data);
    }
}
