<?php

namespace App\Controllers;

use App\Models\SalesPaymentModel;
use App\Models\SaleModel;
use App\Models\CustomerModel;
use App\Models\ProductModel;

class SalesPaymentController extends BaseController
{
    protected $paymentModel;
    protected $saleModel;
    protected $customerModel;
    protected $productModel;

    public function __construct()
    {
        $this->paymentModel  = new SalesPaymentModel();
        $this->saleModel     = new SaleModel();
        $this->customerModel = new CustomerModel();
        $this->productModel  = new ProductModel();
    }

    // List all customers with balance and payments
    public function index()
    {
        $customers = $this->customerModel->findAll();

        $salesByCustomer = [];
        foreach ($customers as $customer) {
            $customerSales = $this->saleModel->where('customer_id', $customer['id'])->findAll();

            $totalSales = 0;
            foreach ($customerSales as $sale) {
                $totalSales += $sale['price'] * $sale['quantity'];
            }

            $totalPaid = $this->paymentModel
                ->where('customer_id', $customer['id'])
                ->select('SUM(amount) as total_paid')
                ->first()['total_paid'] ?? 0;

            $balance = $totalSales - $totalPaid;

            $salesByCustomer[] = [
                'customer_id'   => $customer['id'],
                'customer_name' => $customer['name'],
                'balance'       => $balance
            ];
        }

        $payments = $this->paymentModel
            ->select('sales_payments.*, customers.name AS customer_name')
            ->join('customers', 'customers.id = sales_payments.customer_id', 'left')
            ->orderBy('sales_payments.created_at', 'DESC')
            ->findAll();

        return view('salespayment/index', [
            'sales'    => $salesByCustomer,
            'payments' => $payments,
            'success'  => session()->getFlashdata('success'),
            'error'    => session()->getFlashdata('error'),
        ]);
    }

    // Store a new payment
    public function store()
    {
        $post = $this->request->getPost();

        $validation = \Config\Services::validation();
        $validation->setRules([
            'sale_id' => 'required', // actually customer_id
            'amount'  => 'required|numeric',
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        $customerId = $post['sale_id'];

        $customer = $this->customerModel->find($customerId);
        if (!$customer) {
            return redirect()->back()->with('error', 'Customer not found.');
        }

        $customerSales = $this->saleModel->where('customer_id', $customerId)->findAll();
        $totalSales = 0;
        foreach ($customerSales as $s) {
            $totalSales += $s['price'] * $s['quantity'];
        }

        $totalPaid = $this->paymentModel->where('customer_id', $customerId)
                                        ->select('SUM(amount) as total_paid')
                                        ->first()['total_paid'] ?? 0;

        $balance = $totalSales - $totalPaid;

        if ($post['amount'] > $balance) {
            return redirect()->back()->withInput()->with('error', 'Payment exceeds balance!');
        }

        $data = [
            'customer_id'  => $customerId,
            'amount'       => $post['amount'],
            'payment_date' => $post['payment_date'] ?: date('Y-m-d'),
            'note'         => $post['note'] ?? null,
            'created_at'   => date('Y-m-d H:i:s'),
        ];

        $this->paymentModel->insert($data);

        return redirect()->to(site_url('sales-payments'))->with('success', 'Payment added successfully.');
    }

    // Delete a payment
    public function delete($id)
    {
        $this->paymentModel->delete($id);
        return redirect()->to(site_url('sales-payments'))->with('success', 'Payment deleted successfully.');
    }

    // View all sales and payments for a customer
    public function view($customerId = null)
{
    if (!$customerId) {
        return redirect()->to(site_url('sales-payments'))
                         ->with('error', 'Customer not found.');
    }

    // Fetch all sales for this customer
    $sales = $this->saleModel->where('customer_id', $customerId)->findAll();

    if (!$sales) {
        return redirect()->to(site_url('sales-payments'))
                         ->with('error', 'No sales found for this customer.');
    }

    // Aggregate products for the customer
    $saleProducts = [];
    $totalAmount = 0;
    foreach ($sales as $s) {
        $product = $this->productModel->find($s['product_id']);
        $saleProducts[] = [
            'name'     => $product['name'] ?? 'Unknown',
            'quantity' => $s['quantity'],
            'price'    => $s['price'],
            'total'    => $s['price'] * $s['quantity']
        ];
        $totalAmount += $s['price'] * $s['quantity'];
    }

    // Fetch all payments for this customer
    $payments = $this->paymentModel
                     ->where('customer_id', $customerId)
                     ->orderBy('created_at', 'DESC')
                     ->findAll();

    $totalPaid = 0;
    foreach ($payments as $p) {
        $totalPaid += $p['amount'];
    }

    $data = [
        'customer'     => $this->customerModel->find($customerId),
        'saleProducts' => $saleProducts,
        'totalAmount'  => $totalAmount,
        'totalPaid'    => $totalPaid,
        'balance'      => $totalAmount - $totalPaid,
        'salePayments' => $payments
    ];

    return view('salespayment/view', $data);
}

}
