<?php

namespace App\Controllers;

use App\Models\VendorModel;
use CodeIgniter\Controller;
use Dompdf\Dompdf;

class VendorsController extends Controller
{
    protected $vendorModel;

    public function __construct()
    {
        $this->vendorModel = new VendorModel();
        helper(['form', 'url']);
    }

    /**
     * Show vendors list (with search + pagination)
     */
    public function index()
    {
        if (!session()->get('logged_in')) {
            return redirect()->to('/login');
        }

        $search = $this->request->getGet('search');

        $builder = $this->vendorModel;

        if (!empty($search)) {
            $builder = $builder
                ->groupStart()
                ->like('name', $search)
                ->orLike('phone', $search)
                ->groupEnd();
        }

        $vendors = $builder->orderBy('id', 'DESC')->paginate(10); // 10 per page

        $data = [
            'vendors' => $vendors,
            'pager'   => $this->vendorModel->pager,
            'search'  => $search,
            'success' => session()->getFlashdata('success'),
            'errors'  => session()->getFlashdata('errors'),
        ];

        return view('vendors/index', $data);
    }

    /**
     * Store new vendor
     */
    public function store()
    {
        $validation = \Config\Services::validation();
        $validation->setRules([
            'name' => 'required|min_length[2]|max_length[150]',
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()
                ->to('/vendors')
                ->with('errors', $validation->getErrors())
                ->withInput();
        }

        $this->vendorModel->insert([
            'name'    => $this->request->getPost('name'),
            'phone'   => $this->request->getPost('phone'),
            'address' => $this->request->getPost('address'),
        ]);

        return redirect()->to('/vendors')->with('success', 'Vendor added successfully.');
    }

    /**
     * Delete vendor
     */
    public function delete($id)
    {
        $vendor = $this->vendorModel->find($id);

        if (!$vendor) {
            return redirect()->to('/vendors')->with('errors', ['Vendor not found.']);
        }

        $this->vendorModel->delete($id);

        return redirect()->to('/vendors')->with('success', 'Vendor deleted successfully.');
    }

    /**
     * View vendor details (AJAX)
     */
    public function view($id)
    {
        $vendor = $this->vendorModel->find($id);

        if (!$vendor) {
            return $this->response
                ->setStatusCode(404)
                ->setJSON(['error' => 'Vendor not found']);
        }

        return $this->response->setJSON([
            'id'         => $vendor['id'],
            'name'       => $vendor['name'],
            'phone'      => $vendor['phone'] ?? '',
            'address'    => $vendor['address'] ?? '',
            'created_at' => $vendor['created_at'] ?? date('Y-m-d H:i:s'),
            'updated_at' => $vendor['updated_at'] ?? date('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Download vendors as PDF
     */
    public function download()
    {
        $vendors = $this->vendorModel->orderBy('id', 'DESC')->findAll();

        // Start building HTML for PDF
        $html = '<h2 style="text-align:center;">Vendors List</h2>';
        $html .= '<table width="100%" border="1" cellspacing="0" cellpadding="5" style="border-collapse: collapse;">';
        $html .= '<thead>
                    <tr>
                        <th>#</th>
                        <th>Name</th>
                        <th>Phone</th>
                        <th>Address</th>
                    </tr>
                  </thead>';
        $html .= '<tbody>';
        foreach ($vendors as $index => $vendor) {
            $html .= '<tr>
                        <td>'.($index + 1).'</td>
                        <td>'.$vendor['name'].'</td>
                        <td>'.($vendor['phone'] ?? '-').'</td>
                        <td>'.($vendor['address'] ?? '-').'</td>
                      </tr>';
        }
        $html .= '</tbody></table>';

        // Initialize Dompdf
        $dompdf = new Dompdf();
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        // Output PDF to browser
        $dompdf->stream('vendors_list.pdf', ['Attachment' => true]);
        exit;
    }
}
