<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Sales Payments</title>

<script src="https://cdn.tailwindcss.com"></script>

<link href="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/css/select2.min.css" rel="stylesheet" />
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/select2.min.js"></script>
</head>

<body class="bg-gray-100 min-h-screen">

<?= view('header'); ?>

<div class="max-w-7xl mx-auto px-6 mt-6">

    <!-- Back -->
    <div class="mb-4">
        <a href="<?= site_url('dashboard') ?>"
           class="bg-gray-600 text-white px-4 py-2 rounded hover:bg-gray-700">
            ← Back
        </a>
    </div>

    <!-- Title -->
    <div class="mb-6">
        <h1 class="text-3xl font-bold text-gray-800">Sales Payments</h1>
        <p class="text-gray-500">Record payments for customer sales</p>
    </div>

    <!-- Payment Form -->
    <div class="bg-white rounded-xl shadow p-6 mb-6">
        <h2 class="text-lg font-semibold mb-4">Record New Payment</h2>

        <form id="paymentForm" class="grid grid-cols-1 md:grid-cols-4 gap-4">

            <!-- Sale -->
            <div>
                <label class="block text-sm mb-1">Sale</label>
                <select id="sale_id" class="select2 w-full" required>
                    <option value="">Select Sale</option>
                    <?php foreach ($sales as $sale): ?>
                        <option
                            value="<?= $sale['id'] ?>"
                            data-balance="<?= (float)$sale['balance'] ?>"
                            data-paid="<?= (float)$sale['amount_paid'] ?>">
                            Sale #<?= $sale['id'] ?> – <?= esc($sale['customer_name']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <!-- Product -->
            <div>
                <label class="block text-sm mb-1">Product</label>
                <select id="product_id" class="select2 w-full" required>
                    <option value="">Select sale first</option>
                </select>
            </div>

            <!-- Amount -->
            <div>
                <label class="block text-sm mb-1">Amount</label>
                <input
                    type="number"
                    step="0.01"
                    id="amount"
                    class="w-full p-2 border rounded"
                    required>
            </div>

            <!-- Submit -->
            <div class="flex items-end">
                <button
                    type="submit"
                    class="w-full bg-green-500 text-white py-2 rounded hover:bg-green-600">
                    Record Payment
                </button>
            </div>

        </form>

        <p id="balance_info" class="mt-3 text-sm text-gray-600">
            Select a sale to view balance.
        </p>
    </div>

    <!-- Payments Table -->
    <div class="bg-white rounded-xl shadow p-4 overflow-x-auto">
        <table class="min-w-full border">
            <thead class="bg-green-50">
                <tr>
                    <th class="border px-3 py-2">#</th>
                    <th class="border px-3 py-2">Customer</th>
                    <th class="border px-3 py-2">Product</th>
                    <th class="border px-3 py-2">Sale Total</th>
                    <th class="border px-3 py-2">Paid</th>
                    <th class="border px-3 py-2">Date</th>
                </tr>
            </thead>
            <tbody>
                <?php if ($payments): ?>
                    <?php foreach ($payments as $p): ?>
                        <tr class="hover:bg-green-50">
                            <td class="border px-3 py-2"><?= $p['id'] ?></td>
                            <td class="border px-3 py-2"><?= esc($p['customer_name']) ?></td>
                            <td class="border px-3 py-2"><?= esc($p['product_name']) ?></td>
                            <td class="border px-3 py-2">$<?= number_format($p['sale_total'],2) ?></td>
                            <td class="border px-3 py-2 text-green-600 font-semibold">
                                $<?= number_format($p['amount'],2) ?>
                            </td>
                            <td class="border px-3 py-2">
                                <?= date('Y-m-d', strtotime($p['created_at'])) ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="6" class="text-center p-4 text-gray-500">
                            No payments recorded
                        </td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

</div>

<script>
$(function () {

    $('.select2').select2({ width: '100%' });

    const saleSelect    = $('#sale_id');
    const productSelect = $('#product_id');
    const amountInput   = $('#amount');
    const balanceInfo   = $('#balance_info');

    // When sale changes
    saleSelect.on('change', function () {

        const opt = $(this).find(':selected');
        const saleId  = $(this).val();
        const paid    = parseFloat(opt.data('paid')) || 0;
        const balance = parseFloat(opt.data('balance')) || 0;

        productSelect.html('<option value="">Loading...</option>').trigger('change');
        amountInput.val('');

        if (!saleId) {
            balanceInfo.text('Select a sale to view balance.');
            return;
        }

        balanceInfo.text(
            `Amount Paid: $${paid.toFixed(2)} | Remaining: $${balance.toFixed(2)}`
        );

        $.ajax({
            url: "<?= site_url('sales/getSaleProducts') ?>/" + saleId,
            dataType: "json",
            success: function (products) {

                productSelect.empty().append(
                    '<option value="">Select Product</option>'
                );

                if (products.length === 0) {
                    productSelect.append(
                        '<option value="">No unpaid products</option>'
                    );
                } else {
                    products.forEach(p => {
                        productSelect.append(
                            `<option value="${p.id}" data-balance="${p.balance}">
                                ${p.name} (Balance: $${p.balance})
                            </option>`
                        );
                    });
                }

                productSelect.trigger('change');
            },
            error: function () {
                balanceInfo.text('❌ Error loading products');
                productSelect.html('<option value="">Error</option>')
                    .trigger('change');
            }
        });
    });

    // Product change
    productSelect.on('change', function () {
        const bal = parseFloat($(this).find(':selected').data('balance')) || 0;
        amountInput.attr('max', bal);
        if (bal > 0) {
            balanceInfo.text(`Selected product balance: $${bal.toFixed(2)}`);
        }
    });

    // Submit
    $('#paymentForm').on('submit', function (e) {
        e.preventDefault();

        $.post(
            "<?= site_url('sales/addPayment') ?>",
            {
                sale_id: saleSelect.val(),
                product_id: productSelect.val(),
                amount: amountInput.val()
            },
            () => location.reload()
        );
    });

});
</script>

</body>
</html>
